/*
 * The MIT License (MIT)

 * Copyright (c) 2025 GenText-Checker Developers

 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "common.h"
#include "min_edit_distance.h"
#include "cosine_similarity.h"
#include "cxxtoken.h"

#include <iostream>
#include <vector>
#include <cstring> 

using std::string;
using std::vector;

const std::string argv_error_msg = "Please input right arguments: e.g., ./gtchecker file_a file_b";

int main(int argc, char** argv) {
  if (argc < 3) {
    std::cerr << argv_error_msg << std::endl;
    return -1;
  } else {
    string file_a = argv[1];
    string file_b = argv[2];

    // Read TXT file into string
    string document_A = gtchecker::ReadFileToString(file_a);
    string document_B = gtchecker::ReadFileToString(file_b);

    if (document_A.empty() || document_B.empty()) {
      return -1;
    }

    // Split string by blank spaces
    vector<string> document_A_words = gtchecker::GetSplitWords(document_A);
    vector<string> document_B_words = gtchecker::GetSplitWords(document_B);

    double similarity = 0.0;

    // On default, gtchecker will selcect similarity calculator automatically
    if (argc == 3) {
      // Compare the size of two strings
      // For instance, document_a = "abcd", size = 4
      //               document_b = "abcde", size = 5
      // diff_ratio = (5-4) / 4 = 0.25
      double diff_ratio = gtchecker::SizeDifferenceRatio(document_A, 
                                                         document_B);
      // If the size difference of two doucments is not very big
      if (diff_ratio < 0.2) {
        gtchecker::CosineSimilarity cosine_similarity(document_A_words,
                                                      document_B_words);
        similarity = cosine_similarity.CalculateSimilarity();
      } else {
        gtchecker::MEDSimilarity med_similarity(document_A_words,
                                                document_B_words);
        similarity = med_similarity.CalculateSimilarity();
      }
    } else { // User choose "--med" or "--cosine" similarity calculator by hand
      if (strcmp(argv[3], "--med") == 0) {
        gtchecker::MEDSimilarity med_similarity(document_A_words,
                                                document_B_words);
        similarity = med_similarity.CalculateSimilarity();
      } else if (strcmp(argv[3], "--cosine") == 0) {
        gtchecker::CosineSimilarity cosine_similarity(document_A_words,
                                                      document_B_words);
        similarity = cosine_similarity.CalculateSimilarity();
      } else {
        std::cerr << argv_error_msg << std::endl;
        return -1;
      }
    }

    std::cout << "The similarity of " << file_a << " and " 
              << file_b << " is: " << similarity << std::endl;
  }

  return 0;
}