/*
 * The MIT License (MIT)

 * Copyright (c) 2025 GenText-Checker Developers

 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "common.h"
#include "min_edit_distance.h"
#include "cosine_similarity.h"
#include "cxxtoken.h"

#include <iostream>
#include <vector>
#include <cmath>

using std::string;
using std::vector;

int TestMEDSimilarity(int* failed_case, vector<int>* failed_case_number) {
  int total_case = 5;

  // case 1
  vector<string> array_a = {"1", "2", "3", "4"};
  vector<string> array_b = {"1", "2", "3", "4", "5"};

  gtchecker::MEDSimilarity MED_1(array_a, array_b);

  double similarity = MED_1.CalculateSimilarity();

  if (similarity != 1.0) {
    (*failed_case)++;
    failed_case_number->push_back(1);
  }

  // case 2
  array_a = {"1", "2", "3", "4"};
  array_b = {"1", "1", "3", "4"};

  gtchecker::MEDSimilarity MED_2(array_a, array_b);

  similarity = MED_2.CalculateSimilarity();

  if (similarity != 0.75) {
    (*failed_case)++;
    failed_case_number->push_back(2);
  }

  // case 3
  array_a = {"1", "2", "3", "4"};
  array_b = {"2", "4"};

  gtchecker::MEDSimilarity MED_3(array_a, array_b);

  similarity = MED_3.CalculateSimilarity();

  if (similarity != 1.0) {
    (*failed_case)++;
    failed_case_number->push_back(3);
  }

  // case 4
  array_a = {"1", "2", "3", "4"};
  array_b = {"2", "4", "5"};

  gtchecker::MEDSimilarity MED_4(array_a, array_b);

  similarity = MED_4.CalculateSimilarity();

  if (similarity != 0.5) {
    (*failed_case)++;
    failed_case_number->push_back(4);
  }

  // case 5
  array_a = {"1", "2", "3", "4"};
  array_b = {"2", "4", "4"};

  gtchecker::MEDSimilarity MED_5(array_a, array_b);

  similarity = MED_5.CalculateSimilarity();

  if (similarity != 0.75) {
    (*failed_case)++;
    failed_case_number->push_back(5);
  }

  return total_case;
}

int TestCosineSimilarity(int* failed_case, vector<int>* failed_case_number) {
  int total_case = 5;

  // case 6
  vector<string> array_a = {"1", "2", "3", "4"};
  vector<string> array_b = {"1", "2", "3", "4", "5"};

  gtchecker::CosineSimilarity COS_1(array_a, array_b);

  double similarity = COS_1.CalculateSimilarity();

  if (similarity != 2.0/std::sqrt(5)) {
    (*failed_case)++;
    failed_case_number->push_back(6);
  }

  // case 7
  array_a = {"1", "2", "3", "4"};
  array_b = {"1", "1", "3", "4"};

  gtchecker::CosineSimilarity COS_2(array_a, array_b);

  similarity = COS_2.CalculateSimilarity();

  if (similarity != 2.0/std::sqrt(6)) {
    (*failed_case)++;
    failed_case_number->push_back(7);
  }

  // case 8
  array_a = {"1", "2", "3", "4"};
  array_b = {"2", "4"};

  gtchecker::CosineSimilarity COS_3(array_a, array_b);

  similarity = COS_3.CalculateSimilarity();

  if (similarity != 1.0/std::sqrt(2)) {
    (*failed_case)++;
    failed_case_number->push_back(8);
  }

  // case 9
  array_a = {"1", "2", "3", "4"};
  array_b = {"2", "4", "5"};

  gtchecker::CosineSimilarity COS_4(array_a, array_b);

  similarity = COS_4.CalculateSimilarity();

  if (similarity != 1.0/std::sqrt(3)) {
    (*failed_case)++;
    failed_case_number->push_back(9);
  }

  // case 10
  array_a = {"1", "2", "3", "4"};
  array_b = {"2", "4", "4"};

  gtchecker::CosineSimilarity COS_5(array_a, array_b);

  similarity = COS_5.CalculateSimilarity();

  if (similarity != 3.0/(2*std::sqrt(5))) {
    (*failed_case)++;
    failed_case_number->push_back(10);
  }

  return total_case;
}

int TestCXXToken(int* failed_case, vector<int>* failed_case_number) {
  int total_case = 4;

  // case 11
  string doc_1 = "Hello World";
  vector<string> vec_1 = gtchecker::GetSplitWords(doc_1);
  if (vec_1[0] != "hello" || vec_1[1] != "world") {
    (*failed_case)++;
    failed_case_number->push_back(11);
  }

  // case 12
  string doc_2 = "Hello World.";
  vector<string> vec_2 = gtchecker::GetSplitWords(doc_2);
  if (vec_2[0] != "hello" || vec_2[1] != "world") {
    (*failed_case)++;
    failed_case_number->push_back(12);
  }

  // case 13
  string doc_3 = "Hello World...";
  vector<string> vec_3 = gtchecker::GetSplitWords(doc_3);
  if (vec_3[0] != "hello" || vec_3[1] != "world") {
    (*failed_case)++;
    failed_case_number->push_back(13);
  }

  // case 14
  string doc_4 = "..1(Hello.. --World...";
  vector<string> vec_4 = gtchecker::GetSplitWords(doc_3);
  if (vec_4[0] != "hello" || vec_4[1] != "world") {
    (*failed_case)++;
    failed_case_number->push_back(14);
  }

  return total_case;
}

int TestSizeDifferentRatio(int* failed_case, vector<int>* failed_case_number) {
  int total_case = 3;

  // case 15
  string str_a = "1234";
  string str_b = "1234";
  double ratio = gtchecker::SizeDifferenceRatio(str_a, str_b);
  if (ratio != 0) {
    (*failed_case)++;
    failed_case_number->push_back(15);
  }

  // case 16
  str_a = "12345";
  str_b = "1234";
  ratio = gtchecker::SizeDifferenceRatio(str_a, str_b);
  if (ratio != 0.2) {
    (*failed_case)++;
    failed_case_number->push_back(16);
  }

  // case 17
  str_a = "1234";
  str_b = "12345";
  ratio = gtchecker::SizeDifferenceRatio(str_a, str_b);
  if (ratio != 0.25) {
    (*failed_case)++;
    failed_case_number->push_back(17);
  }

  return total_case;
}

int main(int argc, char** argv) {
  int total_case = 0;
  int failed_case = 0;
  vector<int> failed_case_number;

  total_case += TestMEDSimilarity(&failed_case, &failed_case_number);
  total_case += TestCosineSimilarity(&failed_case, &failed_case_number);
  total_case += TestCXXToken(&failed_case, &failed_case_number);
  total_case += TestSizeDifferentRatio(&failed_case, &failed_case_number);

  std::cout << total_case - failed_case << " cases passed. " 
            << failed_case << " cases failed." << std::endl;

  if (failed_case_number.size() != 0) {
    std::cout << "Failed cases: " << std::endl;
    for (int i = 0; i < failed_case_number.size(); ++i) {
      std::cout << "case " << failed_case_number[i] << std::endl;
    }
  }

  return 0;
}