/*
 * The MIT License (MIT)

 * Copyright (c) 2025 GenText-Checker Developers

 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "common.h"
#include "spliter.h"
#include "args_parser.h"
#include "similarity.h"
#include "tokenizer.h"
#include "signature.h"

#include <vector>
#include <string>
#include <iostream>

int TestSpliter(int* failed_cases, std::vector<int>* failed_cases_number) {
  int total_cases = 6;

  // case 1
  std::string doc_A = gtchecker::ReadFileToString("./doc_A.txt");
  gtchecker::Spliter spliter_1(doc_A);
  std::vector<gtchecker::Sentence> result_1 = spliter_1.split();

  if (result_1.size() != 11) {
    (*failed_cases)++;
    failed_cases_number->push_back(1);
  }

  // case 2
  std::string doc_B = gtchecker::ReadFileToString("./doc_B.txt");
  gtchecker::Spliter spliter_2(doc_B);
  std::vector<gtchecker::Sentence> result_2 = spliter_2.split();

  if (result_2.size() != 11) {
    (*failed_cases)++;
    failed_cases_number->push_back(2);
  }

  // case 3
  gtchecker::Spliter spliter_3("a b c d");
  std::vector<gtchecker::Sentence> result_3 = spliter_3.split();

  if (result_3.size() != 1) {
    (*failed_cases)++;
    failed_cases_number->push_back(3);
  }

  // case 4
  gtchecker::Spliter spliter_4("   a.. b c d  ");
  std::vector<gtchecker::Sentence> result_4 = spliter_4.split();

  if (result_4.size() != 2) {
    (*failed_cases)++;
    failed_cases_number->push_back(4);
  }

  // case 5
  gtchecker::Spliter spliter_5(".   a. b#.  c d  ");
  std::vector<gtchecker::Sentence> result_5 = spliter_5.split();

  if (result_5.size() != 3) {
    (*failed_cases)++;
    failed_cases_number->push_back(5);
  }

  // case 6
  gtchecker::Spliter spliter_6(".   a. b#.  c d  4 dd ./ 33 a ");
  std::vector<gtchecker::Sentence> result_6 = spliter_6.split();

  if (result_6.size() != 3 || result_6[2].sentence() != "c d  4 dd ./ 33 a ") {
    (*failed_cases)++;
    failed_cases_number->push_back(6);
  }

  return total_cases;
}

int TestArgsParser(int* failed_cases, std::vector<int>* failed_cases_number) {
  int total_cases = 10;

  // case 7
  const char* strings_1[] = {"./gtchecker", "plag", "doc_1.txt", "doc_2.txt"};
  size_t size_1 = std::size(strings_1);
  char** prtArray_1 = const_cast<char**>(strings_1);
  gtchecker::ArgsParser parser_1(size_1, prtArray_1);

  if (parser_1.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_number->push_back(7);
  }

  // case 8
  const char* strings_2[] = {"./gtchecker", "plag", "doc_1.txt", "doc_2.txt", "--strict"};
  size_t size_2 = std::size(strings_2);
  char** prtArray_2 = const_cast<char**>(strings_2);
  gtchecker::ArgsParser parser_2(size_2, prtArray_2);

  if (parser_2.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_number->push_back(8);
  }

  // case 9
  const char* strings_3[] = {"./gtchecker", "plag", "doc_1.txt", "doc_2.txt", "--loose"};
  size_t size_3 = std::size(strings_3);
  char** prtArray_3 = const_cast<char**>(strings_3);
  gtchecker::ArgsParser parser_3(size_3, prtArray_3);
  if (parser_3.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_number->push_back(9);
  }

  // case 10
  const char* strings_4[] = {"./gtchecker", "sig", "doc_1.txt"};
  size_t size_4 = std::size(strings_4);
  char** prtArray_4 = const_cast<char**>(strings_4);
  gtchecker::ArgsParser parser_4(size_4, prtArray_4);
  if (parser_4.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_number->push_back(10);
  }

  // case 11
  const char* strings_5[] = {"./gtchecker", "sig_compare", "doc_1.txt", "doc_2.txt"};
  size_t size_5 = std::size(strings_5);
  char** prtArray_5 = const_cast<char**>(strings_5);
  gtchecker::ArgsParser parser_5(size_5, prtArray_5);
  if (parser_5.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_number->push_back(11);
  }


  // case 12
  const char* strings_6[] = {"./gtchecker", "doc_1.txt"};
  size_t size_6 = std::size(strings_6);
  char** prtArray_6 = const_cast<char**>(strings_6);
  gtchecker::ArgsParser parser_6(size_6, prtArray_6);
  if (parser_6.ArgsCheck() == true) {
    (*failed_cases)++;
    failed_cases_number->push_back(12);
  }

  // case 13
  const char* strings_7[] = {"./gtchecker", "exe", "doc_1.txt"};
  size_t size_7 = std::size(strings_7);
  char** prtArray_7 = const_cast<char**>(strings_7);
  gtchecker::ArgsParser parser_7(size_7, prtArray_7);
  if (parser_7.ArgsCheck() == true) {
    (*failed_cases)++;
    failed_cases_number->push_back(13);
  }

  // case 14
  const char* strings_8[] = {"./gtchecker", "plag", "doc_1.txt"};
  size_t size_8 = std::size(strings_8);
  char** prtArray_8 = const_cast<char**>(strings_8);
  gtchecker::ArgsParser parser_8(size_8, prtArray_8);
  if (parser_8.ArgsCheck() == true) {
    (*failed_cases)++;
    failed_cases_number->push_back(14);
  }

  // case 15
  const char* strings_9[] = {"./gtchecker", "plag", "doc_1.txt", "doc_2.txt", "--cosine"};
  size_t size_9 = std::size(strings_9);
  char** prtArray_9 = const_cast<char**>(strings_9);
  gtchecker::ArgsParser parser_9(size_9, prtArray_9);
  if (parser_9.ArgsCheck() == true) {
    (*failed_cases)++;
    failed_cases_number->push_back(15);
  }

  // case 16
  const char* strings_10[] = {"./gtchecker", "sig_compare", "doc_1.txt"};
  size_t size_10 = std::size(strings_10);
  char** prtArray_10 = const_cast<char**>(strings_10);
  gtchecker::ArgsParser parser_10(size_10, prtArray_10);
  if (parser_10.ArgsCheck() == true) {
    (*failed_cases)++;
    failed_cases_number->push_back(16); 
  }

  return total_cases;
}

int TestSimilarity(int* failed_cases, std::vector<int>* failed_cases_number) {
  int total_cases = 2;

  // case 17
  std::string doc_A = gtchecker::ReadFileToString("./doc_A.txt");
  std::string doc_B = gtchecker::ReadFileToString("./doc_B.txt");

  gtchecker::Spliter spliter_A(doc_A);
  gtchecker::Spliter spliter_B(doc_B);

  std::vector<gtchecker::Sentence> chunks_A = spliter_A.split();
  std::vector<gtchecker::Sentence> chunks_B = spliter_B.split();

  gtchecker::Similarity loose_similarity(chunks_A, chunks_B, "--loose");

  std::vector<std::vector<double>> similarity_results = loose_similarity.Calculate();

  int similar_sentence = 0;

  for (int i = 0; i < chunks_A.size(); ++i) {
    std::vector<double> tmp = similarity_results[i];
    for (int j = 0; j < chunks_B.size(); ++j) {
      if (tmp[j] == 1.0) {
        similar_sentence++;
      }
    }
  }

  if (similar_sentence != 4) {
    (*failed_cases)++;
    failed_cases_number->push_back(17);
  }

  // case 18
  gtchecker::Similarity strict_similarity(chunks_A, chunks_B, "--strict");

  similarity_results = strict_similarity.Calculate();

  similar_sentence = 0;

  for (int i = 0; i < chunks_A.size(); ++i) {
    std::vector<double> tmp = similarity_results[i];
    for (int j = 0; j < chunks_B.size(); ++j) {
      if (tmp[j] == 1.0) {
        similar_sentence++;
      }
    }
  }

  if (similar_sentence != 11) {
    (*failed_cases)++;
    failed_cases_number->push_back(18);
  }

  return total_cases;
}

int TestTokenizer(int* failed_cases, std::vector<int>* failed_cases_number) {
  int total_case = 4;

  // case 19
  std::string doc_1 = "Hello World";
  gtchecker::Tokenizer tokenizer;
  std::vector<std::string> vec_1 = tokenizer.GetSplitWords(doc_1);
  if (vec_1[0] != "hello" || vec_1[1] != "world") {
    (*failed_cases)++;
    failed_cases_number->push_back(19);
  }

  // case 20
  std::string doc_2 = "Hello World.";
  std::vector<std::string> vec_2 = tokenizer.GetSplitWords(doc_2);
  if (vec_2[0] != "hello" || vec_2[1] != "world") {
    (*failed_cases)++;
    failed_cases_number->push_back(20);
  }

  // case 21
  std::string doc_3 = "Hello World...";
  std::vector<std::string> vec_3 = tokenizer.GetSplitWords(doc_3);
  if (vec_3[0] != "hello" || vec_3[1] != "world") {
    (*failed_cases)++;
    failed_cases_number->push_back(21);
  }

  // case 22
  std::string doc_4 = "..1(Hello.. --World...";
  std::vector<std::string> vec_4 = tokenizer.GetSplitWords(doc_4);
  if (vec_4[0] != "1(hello" || vec_4[1] != "world") {
    (*failed_cases)++;
    failed_cases_number->push_back(22);
  }

  return total_case;
}

int TestSignature(int* failed_cases, std::vector<int>* failed_cases_number) {
  // case 23
  int total_cases = 1;

  std::string doc_1 = "sah61 i hgta to 11n was dh i at 232 do. all oppp from  o and she. had by.";

  gtchecker::Signature signature(doc_1);

  if (signature.sign() != "4374gTks1opK") {
    (*failed_cases)++;
    failed_cases_number->push_back(23);
  }

  return total_cases;
}

int main(int argc, char** argv) {
  int total_cases = 0;
  int failed_cases = 0;
  std::vector<int> failed_cases_number;

  total_cases += TestSpliter(&failed_cases, &failed_cases_number);
  total_cases += TestArgsParser(&failed_cases, &failed_cases_number);
  total_cases += TestTokenizer(&failed_cases, &failed_cases_number);
  total_cases += TestSimilarity(&failed_cases, &failed_cases_number);
  total_cases += TestSignature(&failed_cases, &failed_cases_number);
  
  std::cout << total_cases - failed_cases << " cases passed. " 
            << failed_cases << " cases failed." << std::endl;

  if (failed_cases_number.size() != 0) {
    std::cout << "Failed cases: " << std::endl;
    for (int i = 0; i < failed_cases_number.size(); ++i) {
      std::cout << "case " << failed_cases_number[i] << std::endl;
    }
  }
}