/*
 * The MIT License (MIT)

 * Copyright (c) 2025 GenText-Checker Developers

 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef GTCHECKER_TOKENIZER_H_
#define GTCHECKER_TOKENIZER_H_

#include <string>
#include <iostream>
#include <vector>
#include <cctype>

namespace gtchecker {

class Tokenizer {
 public:
  Tokenizer(const char delim = ' ') 
    : delim_(delim) { }

  std::vector<std::string> GetSplitWords(const std::string& document) {
    std::vector<std::string> result = SplitStringToVectorUsing(document);
    FilterAlphanumeric(&result);
    Lowercase(&result);
    return result;
  }

 private:
  std::vector<std::string> SplitStringToVectorUsing(const std::string& full) {
    std::vector<std::string> result;

    char c = delim_;

    int start = 0;
    int end = full.size()-1;

    while (start != end) {
      if (full[start] == c) {
        ++start;
      } else {
        while (full[end] == c && start != end) {
          --end;
        }
        break;
      }
    }
    
    int pos = start;
    for (int i = start; i <= end; ++i) {
      if (full[i] == c) {
        std::string str = full.substr(pos, i-pos);
        result.push_back(str);
        while (full[i] == c && i <= end) { 
          ++i;
        }
        pos = i;
      }
    }

    std::string str = full.substr(pos, end-pos+1);
    result.push_back(str);

    return result;
  }

  void FilterAlphanumeric(std::vector<std::string>* result) {
    for (int i = 0; i < result->size(); ++i) {
      if ((*result)[i].empty()) {
        continue;
      }

      // Check first element
      while ((*result)[i].empty() == false && !std::isalnum((*result)[i].front())) {
        (*result)[i].erase((*result)[i].begin());
      }

      // Check last element
      while ((*result)[i].empty() == false && !std::isalnum((*result)[i].back())) {
        (*result)[i].erase((*result)[i].end() - 1);
      }
    }
  }

  void Lowercase(std::vector<std::string>* result) {
    for (int i = 0; i < result->size(); ++i) {
      if ((*result)[i].empty()) {
        continue;
      }

      int str_size = (*result)[i].size();

      for (int j = 0; j < str_size; ++j) {
        char c = (*result)[i][j];
        (*result)[i][j] = std::tolower(c);
      }
    }
  }

  char delim_;
};

} // namespace gtchecker

#endif // GTCHECKER_TOKENIZER_H_