/*
 * The MIT License (MIT)

 * Copyright (c) 2025 GenText-Checker Developers

 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef GTCHECKER_PRINTER_H_
#define GTCHECKER_PRINTER_H_

#include "spliter.h"

#include <vector>
#include <iostream>

namespace gtchecker {

class Printer {
 public:
  Printer(const std::string& doc_A_name,
          const std::string& doc_B_name,
          const std::vector<std::vector<double>>& results, 
          const std::vector<gtchecker::Sentence>& chunks_A,
          const std::vector<gtchecker::Sentence>& chunks_B) 
   : doc_A_name_(doc_A_name),
     doc_B_name_(doc_B_name),
     results_(results), 
     chunks_A_(chunks_A), 
     chunks_B_(chunks_B) { }

  void print() {
    int sentence_count = 0;
    std::cout << "Doc_A: " << doc_A_name_ << "  Doc_B: " << doc_B_name_ << std::endl << std::endl;
    for (int i = 0; i < chunks_A_.size(); ++i) {
      std::vector<double> tmp = results_[i];
      for (int j = 0; j < chunks_B_.size(); ++j) {
        if (tmp[j] == 1.0) {
          sentence_count++;
          std::cout << "Doc_A: [" << chunks_A_[i].start_pos() 
                    << "-" << chunks_A_[i].end_pos() 
                    << "] " << chunks_A_[i].sentence() << std::endl;

          std::cout << "Doc_B: [" << chunks_B_[j].start_pos() 
                    << "-" << chunks_B_[j].end_pos() << "] " 
                    << chunks_B_[j].sentence() << std::endl;
        }
      }
    }

    double similarity_A = (double)sentence_count / chunks_A_.size() * 100;
    double similarity_B = (double)sentence_count / chunks_B_.size() * 100;
    std::cout << std::endl;
    std::cout << "Doc_A has " << similarity_A << "% similarity with Doc_B." << std::endl;
    std::cout << "Doc_B has " << similarity_B << "% similarity with Doc_A." << std::endl;
    std::cout << std::endl << std::endl;
  }

 private:
  std::string doc_A_name_;
  std::string doc_B_name_;
  std::vector<std::vector<double>> results_;
  std::vector<gtchecker::Sentence> chunks_A_;
  std::vector<gtchecker::Sentence> chunks_B_;
};

} // namespace gtchecker

#endif // GTCHECKER_PRINTER_H_