/*
 * The MIT License (MIT)

 * Copyright (c) 2025 GenText-Checker Developers

 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "common.h"
#include "spliter.h"
#include "args_parser.h"
#include "similarity.h"
#include "tokenizer.h"
#include "signature.h"

#include <vector>
#include <string>
#include <iostream>

int TestSpliter(int* failed_cases, std::vector<std::string>* failed_cases_name) {
  int total_cases = 6;

  // case 0
  std::string doc_A = gtchecker::ReadFileToString("./doc_A.txt");
  gtchecker::Spliter spliter_0(doc_A);
  std::vector<gtchecker::Sentence> result_0 = spliter_0.split();

  if (result_0.size() != 11) {
    (*failed_cases)++;
    failed_cases_name->push_back("spliter-case-0");
  }

  // case 1
  std::string doc_B = gtchecker::ReadFileToString("./doc_B.txt");
  gtchecker::Spliter spliter_1(doc_B);
  std::vector<gtchecker::Sentence> result_1 = spliter_1.split();

  if (result_1.size() != 11) {
    (*failed_cases)++;
    failed_cases_name->push_back("spliter-case-1");
  }

  // case 2
  gtchecker::Spliter spliter_2("a b c d");
  std::vector<gtchecker::Sentence> result_2 = spliter_2.split();

  if (result_2.size() != 1) {
    (*failed_cases)++;
    failed_cases_name->push_back("spliter-case-2");
  }

  // case 3
  gtchecker::Spliter spliter_3("   a.. b c d  ");
  std::vector<gtchecker::Sentence> result_3 = spliter_3.split();

  if (result_3.size() != 2) {
    (*failed_cases)++;
    failed_cases_name->push_back("spliter-case-3");
  }

  // case 4
  gtchecker::Spliter spliter_4(".   a. b#.  c d  ");
  std::vector<gtchecker::Sentence> result_4 = spliter_4.split();

  if (result_4.size() != 3) {
    (*failed_cases)++;
    failed_cases_name->push_back("spliter-case-4");
  }

  // case 5
  gtchecker::Spliter spliter_5(".   a. b#.  c d  4 dd ./ 33 a ");
  std::vector<gtchecker::Sentence> result_5 = spliter_5.split();

  if (result_5.size() != 3 || result_5[2].sentence() != "c d  4 dd ./ 33 a ") {
    (*failed_cases)++;
    failed_cases_name->push_back("spliter-case-5");
  }

  return total_cases;
}

int TestArgsParser(int* failed_cases, std::vector<std::string>* failed_cases_name) {
  int total_cases = 10;

  size_t size = 0;
  char** prtArray = nullptr;

  // case 0
  const char* strings_0[] = {"./gtchecker", "plag", "doc_1.txt", "doc_2.txt"};
  size = std::size(strings_0);
  prtArray = const_cast<char**>(strings_0);
  gtchecker::ArgsParser parser_0(size, prtArray);

  if (parser_0.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-0-0");
  }

  if (parser_0.command() != "plag") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-0-1");
  }

  if (parser_0.file_a() != "doc_1.txt") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-0-2");
  }

  if (parser_0.file_b() != "doc_2.txt") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-0-3");
  }

  if (parser_0.check_type() != "--strict") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-0-4");
  }

  // case 1
  const char* strings_1[] = {"./gtchecker", "plag", "doc_1.txt", "doc_2.txt", "--strict"};
  size = std::size(strings_1);
  prtArray = const_cast<char**>(strings_1);
  gtchecker::ArgsParser parser_1(size, prtArray);

  if (parser_1.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-1-0");
  }

  if (parser_1.command() != "plag") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-1-1");
  }

  if (parser_1.file_a() != "doc_1.txt") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-1-2");
  }

  if (parser_1.file_b() != "doc_2.txt") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-1-3");
  }

  if (parser_1.check_type() != "--strict") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-1-4");
  }

  if (parser_1.sig_type() != "unset") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-1-5");
  }

  // case 2
  const char* strings_2[] = {"./gtchecker", "plag", "doc_1.txt", "doc_2.txt", "--loose"};
  size = std::size(strings_2);
  prtArray = const_cast<char**>(strings_2);
  gtchecker::ArgsParser parser_2(size, prtArray);
  if (parser_2.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-2-0");
  }

  if (parser_2.command() != "plag") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-2-1");
  }

  if (parser_2.file_a() != "doc_1.txt") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-2-2");
  }

  if (parser_2.file_b() != "doc_2.txt") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-2-3");
  }

  if (parser_2.check_type() != "--loose") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-2-4");
  }

  if (parser_2.sig_type() != "unset") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-2-5");
  }

  // case 3
  const char* strings_3[] = {"./gtchecker", "sig", "doc_1.txt"};
  size = std::size(strings_3);
  prtArray = const_cast<char**>(strings_3);
  gtchecker::ArgsParser parser_3(size, prtArray);
  if (parser_3.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-3-0");
  }

  if (parser_3.command() != "sig") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-3-1");
  }

  if (parser_3.file_a() != "doc_1.txt") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-3-2");
  }

  if (parser_3.file_b() != "unset") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-3-3");
  }

  if (parser_3.check_type() != "unset") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-3-4");
  }

  if (parser_3.sig_type() != "--hfws") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-3-5");
  }

  // case 4
  const char* strings_4[] = {"./gtchecker", "sig", "doc_1.txt", "--hfws"};
  size = std::size(strings_4);
  prtArray = const_cast<char**>(strings_4);
  gtchecker::ArgsParser parser_4(size, prtArray);
  if (parser_4.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-4-0");
  }

  if (parser_4.command() != "sig") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-4-1");
  }

  if (parser_4.file_a() != "doc_1.txt") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-4-2");
  }

  if (parser_4.file_b() != "unset") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-4-3");
  }

  if (parser_4.check_type() != "unset") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-4-4");
  }

  if (parser_4.sig_type() != "--hfws") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-4-5");
  }

  // case 5
  const char* strings_5[] = {"./gtchecker", "sig", "doc_1.txt", "--24bits"};
  size = std::size(strings_5);
  prtArray = const_cast<char**>(strings_5);
  gtchecker::ArgsParser parser_5(size, prtArray);
  if (parser_5.ArgsCheck() == false) {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-5-0");
  }

  if (parser_5.command() != "sig") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-5-1");
  }

  if (parser_5.file_a() != "doc_1.txt") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-5-2");
  }

  if (parser_5.file_b() != "unset") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-5-3");
  }

  if (parser_5.check_type() != "unset") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-5-4");
  }

  if (parser_5.sig_type() != "--24bits") {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-5-5");
  }

  // case 6
  const char* strings_6[] = {"./gtchecker", "doc_1.txt"}; // command missing
  size = std::size(strings_6);
  prtArray = const_cast<char**>(strings_6);
  gtchecker::ArgsParser parser_6(size, prtArray);
  if (parser_6.ArgsCheck() == true) {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-6-0");
  }

  // case 7
  const char* strings_7[] = {"./gtchecker", "exe", "doc_1.txt"}; // command error
  size = std::size(strings_7);
  prtArray = const_cast<char**>(strings_7);
  gtchecker::ArgsParser parser_7(size, prtArray);
  if (parser_7.ArgsCheck() == true) {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-7");
  }

  // case 8
  const char* strings_8[] = {"./gtchecker", "plag", "doc_1.txt"}; // file_b missing
  size = std::size(strings_8);
  prtArray = const_cast<char**>(strings_8);
  gtchecker::ArgsParser parser_8(size, prtArray);
  if (parser_8.ArgsCheck() == true) {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-8");
  }

  // case 9
  const char* strings_9[] = {"./gtchecker", "plag", "doc_1.txt", "doc_2.txt", "--loosing"}; // --strict or losse
  size = std::size(strings_9);
  prtArray = const_cast<char**>(strings_9);
  gtchecker::ArgsParser parser_9(size, prtArray);
  if (parser_9.ArgsCheck() == true) {
    (*failed_cases)++;
    failed_cases_name->push_back("parser-case-9");
  }

  return total_cases;
}


int TestSimilarity(int* failed_cases, std::vector<std::string>* failed_cases_name) {
  int total_cases = 2;

  // case 0
  std::string doc_A = gtchecker::ReadFileToString("./doc_A.txt");
  std::string doc_B = gtchecker::ReadFileToString("./doc_B.txt");

  gtchecker::Spliter spliter_A(doc_A);
  gtchecker::Spliter spliter_B(doc_B);

  std::vector<gtchecker::Sentence> chunks_A = spliter_A.split();
  std::vector<gtchecker::Sentence> chunks_B = spliter_B.split();

  gtchecker::Similarity loose_similarity(chunks_A, chunks_B, "--loose");

  std::vector<std::vector<double>> similarity_results = loose_similarity.Calculate();

  int similar_sentence = 0;

  for (int i = 0; i < chunks_A.size(); ++i) {
    std::vector<double> tmp = similarity_results[i];
    for (int j = 0; j < chunks_B.size(); ++j) {
      if (tmp[j] == 1.0) {
        similar_sentence++;
      }
    }
  }

  if (similar_sentence != 4) {
    (*failed_cases)++;
    failed_cases_name->push_back("similarity-case-0");
  }

  // case 1
  gtchecker::Similarity strict_similarity(chunks_A, chunks_B, "--strict");

  similarity_results = strict_similarity.Calculate();

  similar_sentence = 0;

  for (int i = 0; i < chunks_A.size(); ++i) {
    std::vector<double> tmp = similarity_results[i];
    for (int j = 0; j < chunks_B.size(); ++j) {
      if (tmp[j] == 1.0) {
        similar_sentence++;
      }
    }
  }

  if (similar_sentence != 11) {
    (*failed_cases)++;
    failed_cases_name->push_back("similarity-case-1");
  }

  return total_cases;
}

int TestTokenizer(int* failed_cases, std::vector<std::string>* failed_cases_name) {
  int total_case = 4;

  // case 0
  std::string doc_0 = "Hello World";
  gtchecker::Tokenizer tokenizer;
  std::vector<std::string> vec_0 = tokenizer.GetSplitWords(doc_0);
  if (vec_0[0] != "hello" || vec_0[1] != "world") {
    (*failed_cases)++;
    failed_cases_name->push_back("tokenizer-case-0");
  }

  // case 1
  std::string doc_1 = "Hello World.";
  std::vector<std::string> vec_1 = tokenizer.GetSplitWords(doc_1);
  if (vec_1[0] != "hello" || vec_1[1] != "world") {
    (*failed_cases)++;
    failed_cases_name->push_back("tokenizer-case-1");
  }

  // case 2
  std::string doc_2 = "Hello World...";
  std::vector<std::string> vec_2 = tokenizer.GetSplitWords(doc_2);
  if (vec_2[0] != "hello" || vec_2[1] != "world") {
    (*failed_cases)++;
    failed_cases_name->push_back("tokenizer-case-2");
  }

  // case 3
  std::string doc_3 = "..1(Hello.. --World...";
  std::vector<std::string> vec_3 = tokenizer.GetSplitWords(doc_3);
  if (vec_3[0] != "1(hello" || vec_3[1] != "world") {
    (*failed_cases)++;
    failed_cases_name->push_back("tokenizer-case-3");
  }

  return total_case;
}

int TestSignature(int* failed_cases, std::vector<std::string>* failed_cases_name) {
  int total_cases = 16;

  std::string doc_0 = "sah61 i hgta to 11n was dh i at 232 do. all oppp from o and she. had by.";

  // case 0
  gtchecker::Signature signature_0(doc_0, "--hfws");

  if (signature_0.sign() != "4374gTks1opK") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-0");
  }

  // case 1
  std::string bits_0 = signature_0.char_to_bits('a');

  if (bits_0 != "01100001") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-1");
  }

  // case 2
  std::string bits_1 = signature_0.char_to_bits('9');

  if (bits_1 != "00111001") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-2");
  }

  // case 3
  std::string bits_2 = signature_0.char_to_bits('A');

  if (bits_2 != "01000001") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-3");
  }

  // case 4
  std::vector<int> sig(24, 0);
  signature_0.concat(&sig, bits_0, 0);
  signature_0.concat(&sig, bits_1, 1);
  signature_0.concat(&sig, bits_2, 2);
  std::vector<int> vec = {-1,1,1,-1,-1,-1,-1,1,-1,-1,1,1,1,-1,-1,1,-1,1,-1,-1,-1,-1,-1,1};
  
  for (int i = 0 ; i < sig.size(); ++i) {
    if (sig[i] != vec[i]) {
      (*failed_cases)++;
      failed_cases_name->push_back("signature-case-4");
      break;
    }
  }

  // case 5
  std::string bits_5 = signature_0.char_to_bits('s');

  if (bits_5 != "01110011") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-5");
  }

  // case 6
  std::string bits_6 = signature_0.char_to_bits('q');

  if (bits_6 != "01110001") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-6");
  }

  // case 7
  std::string bits_7 = signature_0.char_to_bits('7');

  if (bits_7 != "00110111") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-7");
  }

  // case 8
  signature_0.concat(&sig, bits_5, 0);
  signature_0.concat(&sig, bits_6, 1);
  signature_0.concat(&sig, bits_7, 2);
  std::vector<int> vec_2 = {-2,2,2,0,-2,-2,0,2,-2,0,2,2,0,-2,-2,2,-2,0,0,0,-2,0,0,2};

  for (int i = 0 ; i < sig.size(); ++i) {
    if (sig[i] != vec_2[i]) {
      (*failed_cases)++;
      failed_cases_name->push_back("signature-case-8");
      break;
    }
  }

  // case 9
  std::string doc_1 = "abcdef";

  gtchecker::Signature signature_1(doc_1, "--24bits");

  std::string sig_1 = signature_1.sign();

  if (sig_1 != "011000000110000001100010") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-9");
  }

  // case 10
  std::string doc_2 = "abcdefghi";

  gtchecker::Signature signature_2(doc_2, "--24bits");

  std::string sig_2 = signature_2.sign();

  if (sig_2 != "011001010110000001100011") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-10");
  }

  // case 11
  std::string doc_3 = "abcdefghi123";

  gtchecker::Signature signature_3(doc_3, "--24bits");

  std::string sig_3 = signature_3.sign();

  if (sig_3 != "011000010110000001100011") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-11");
  }

  // case 12
  std::string doc_4 = "abcdefghi123456";

  gtchecker::Signature signature_4(doc_4, "--24bits");

  std::string sig_4 = signature_4.sign();

  if (sig_4 != "011001010110000001100011") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-12");
  }

  // case 13
  std::string doc_5 = "abcdefghi123456456";

  gtchecker::Signature signature_5(doc_5, "--24bits");

  std::string sig_5 = signature_5.sign();

  if (sig_5 != "001001000010000000100010") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-13");
  }

  // case 14
  std::string doc_6 = "abcdefghi123456456apple";

  gtchecker::Signature signature_6(doc_6, "--24bits");

  std::string sig_6 = signature_6.sign();

  if (sig_6 != "011001000110000001110010") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-13");
  }

  // case 15
  std::string doc_7 = "abc#defghi 1234564 56apple";

  gtchecker::Signature signature_7(doc_7, "--24bits");

  std::string sig_7 = signature_7.sign();

  if (sig_7 != "011001000110000001110010") {
    (*failed_cases)++;
    failed_cases_name->push_back("signature-case-14");
  }

  return total_cases;
}

int main(int argc, char** argv) {
  int total_cases = 0;
  int failed_cases = 0;
  std::vector<std::string> failed_cases_name;

  total_cases += TestSpliter(&failed_cases, &failed_cases_name);
  total_cases += TestArgsParser(&failed_cases, &failed_cases_name);
  total_cases += TestSimilarity(&failed_cases, &failed_cases_name);
  total_cases += TestTokenizer(&failed_cases, &failed_cases_name);
  total_cases += TestSignature(&failed_cases, &failed_cases_name);
  
  std::cout << total_cases - failed_cases << " cases passed. " 
            << failed_cases << " cases failed." << std::endl;

  if (failed_cases_name.size() != 0) {
    std::cout << "Failed cases: " << std::endl;
    for (int i = 0; i < failed_cases_name.size(); ++i) {
      std::cout << failed_cases_name[i] << std::endl;
    }
  }

  return 0;
}